<?php
/**
 * Test Script for Image Watermark
 * 
 * Usage: 
 * 1. Upload this file to your WordPress root directory
 * 2. Access via: https://yoursite.com/test-watermark.php
 * 3. Follow the instructions
 */

// Load WordPress
require_once('wp-load.php');

// Check if user is admin
if (!current_user_can('manage_options')) {
    die('Access denied. Admin only.');
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>Yo ToWebP - Image Watermark Test</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, sans-serif;
            max-width: 1200px;
            margin: 50px auto;
            padding: 20px;
            background: #f0f0f1;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        h1 {
            color: #1d2327;
            border-bottom: 2px solid #2271b1;
            padding-bottom: 10px;
        }
        h2 {
            color: #2271b1;
            margin-top: 30px;
        }
        .status {
            padding: 12px;
            border-radius: 4px;
            margin: 10px 0;
        }
        .status.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        .status.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        .status.warning {
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        .status.info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        code {
            background: #f1f1f1;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: Monaco, Consolas, monospace;
            font-size: 13px;
        }
        pre {
            background: #1d2327;
            color: #f0f0f1;
            padding: 15px;
            border-radius: 4px;
            overflow-x: auto;
            font-size: 13px;
        }
        .test-group {
            margin: 20px 0;
            padding: 20px;
            background: #f9f9f9;
            border-left: 4px solid #2271b1;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 15px 0;
        }
        table th,
        table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        table th {
            background: #f1f1f1;
            font-weight: 600;
        }
        .button {
            display: inline-block;
            padding: 10px 20px;
            background: #2271b1;
            color: white;
            text-decoration: none;
            border-radius: 4px;
            margin: 5px;
        }
        .button:hover {
            background: #135e96;
        }
        img {
            max-width: 100%;
            height: auto;
            border: 1px solid #ddd;
            margin: 10px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🧪 Yo ToWebP - Image Watermark Diagnostic Test</h1>
        <p>This script will test your image watermark configuration and identify any issues.</p>

        <?php
        // Test 1: Check if plugin is active
        echo '<div class="test-group">';
        echo '<h2>Test 1: Plugin Status</h2>';
        
        if (class_exists('YoToWebP')) {
            echo '<div class="status success">✅ Plugin is active</div>';
        } else {
            echo '<div class="status error">❌ Plugin is NOT active</div>';
            echo '<p>Please activate the Yo ToWebP plugin first.</p>';
            echo '</div></div></body></html>';
            exit;
        }
        echo '</div>';

        // Test 2: Check GD/Imagick
        echo '<div class="test-group">';
        echo '<h2>Test 2: Image Libraries</h2>';
        
        $has_gd = extension_loaded('gd');
        $has_imagick = extension_loaded('imagick');
        
        if ($has_gd) {
            echo '<div class="status success">✅ GD Library is available</div>';
            $gd_info = gd_info();
            echo '<table>';
            echo '<tr><th>GD Feature</th><th>Status</th></tr>';
            foreach ($gd_info as $key => $value) {
                $status = $value ? '✓' : '✗';
                echo "<tr><td>$key</td><td>$status</td></tr>";
            }
            echo '</table>';
        } else {
            echo '<div class="status error">❌ GD Library is NOT available</div>';
        }
        
        if ($has_imagick) {
            echo '<div class="status success">✅ Imagick is available</div>';
        } else {
            echo '<div class="status warning">⚠️ Imagick is NOT available (optional)</div>';
        }
        echo '</div>';

        // Test 3: Check watermark settings
        echo '<div class="test-group">';
        echo '<h2>Test 3: Watermark Settings</h2>';
        
        $watermark_enabled = get_option('yo_towebp_watermark_enabled', 0);
        $watermark_type = get_option('yo_towebp_watermark_type', 'text');
        $watermark_image = get_option('yo_towebp_watermark_image', '');
        
        echo '<table>';
        echo '<tr><th>Setting</th><th>Value</th></tr>';
        echo '<tr><td>Watermark Enabled</td><td>' . ($watermark_enabled ? '✅ Yes' : '❌ No') . '</td></tr>';
        echo '<tr><td>Watermark Type</td><td>' . esc_html($watermark_type) . '</td></tr>';
        echo '<tr><td>Watermark Image URL</td><td>' . ($watermark_image ? esc_html($watermark_image) : '(empty)') . '</td></tr>';
        echo '<tr><td>Size</td><td>' . get_option('yo_towebp_watermark_size', 3) . '%</td></tr>';
        echo '<tr><td>Size Type</td><td>' . get_option('yo_towebp_watermark_size_type', 'width') . '</td></tr>';
        echo '<tr><td>Opacity</td><td>' . get_option('yo_towebp_watermark_opacity', 70) . '</td></tr>';
        echo '<tr><td>Position</td><td>' . get_option('yo_towebp_watermark_position', 'bottom-right') . '</td></tr>';
        echo '</table>';
        
        if (!$watermark_enabled) {
            echo '<div class="status warning">⚠️ Watermark is disabled. Enable it in Settings > Yo ToWebP</div>';
        }
        
        if ($watermark_type === 'image' && empty($watermark_image)) {
            echo '<div class="status error">❌ Image watermark selected but no image uploaded!</div>';
            echo '<p>Please upload a logo via Settings > Yo ToWebP > Upload Image/Logo</p>';
        }
        echo '</div>';

        // Test 4: URL to Path Conversion
        if ($watermark_type === 'image' && !empty($watermark_image)) {
            echo '<div class="test-group">';
            echo '<h2>Test 4: URL to Path Conversion</h2>';
            
            echo '<p><strong>Original URL:</strong></p>';
            echo '<code>' . esc_html($watermark_image) . '</code>';
            
            // Simulate url_to_path conversion
            $upload_dir = wp_upload_dir();
            $base_url = $upload_dir['baseurl'];
            $base_dir = $upload_dir['basedir'];
            
            echo '<p><strong>Upload Base URL:</strong></p>';
            echo '<code>' . esc_html($base_url) . '</code>';
            
            echo '<p><strong>Upload Base Directory:</strong></p>';
            echo '<code>' . esc_html($base_dir) . '</code>';
            
            $watermark_path = str_replace($base_url, $base_dir, $watermark_image);
            
            echo '<p><strong>Converted Path:</strong></p>';
            echo '<code>' . esc_html($watermark_path) . '</code>';
            
            // Check if file exists
            if (file_exists($watermark_path)) {
                echo '<div class="status success">✅ File exists at path</div>';
                
                // Check if readable
                if (is_readable($watermark_path)) {
                    echo '<div class="status success">✅ File is readable</div>';
                } else {
                    echo '<div class="status error">❌ File is NOT readable. Check permissions.</div>';
                    echo '<p>Run: <code>chmod 644 ' . esc_html($watermark_path) . '</code></p>';
                }
                
                // Check file size
                $filesize = filesize($watermark_path);
                $filesize_kb = round($filesize / 1024, 2);
                echo '<div class="status info">📦 File size: ' . $filesize_kb . ' KB</div>';
                
                if ($filesize_kb > 500) {
                    echo '<div class="status warning">⚠️ File is larger than 500KB. Consider compressing.</div>';
                }
                
                // Check image info
                $image_info = @getimagesize($watermark_path);
                if ($image_info) {
                    echo '<div class="status success">✅ Valid image file</div>';
                    echo '<table>';
                    echo '<tr><th>Property</th><th>Value</th></tr>';
                    echo '<tr><td>Width</td><td>' . $image_info[0] . 'px</td></tr>';
                    echo '<tr><td>Height</td><td>' . $image_info[1] . 'px</td></tr>';
                    echo '<tr><td>Type</td><td>' . image_type_to_mime_type($image_info[2]) . '</td></tr>';
                    echo '</table>';
                    
                    // Show preview
                    echo '<p><strong>Preview:</strong></p>';
                    echo '<img src="' . esc_url($watermark_image) . '" style="max-width: 300px; background: repeating-conic-gradient(#e0e0e0 0% 25%, #ffffff 0% 50%) 50% / 20px 20px;">';
                    
                    // Check if PNG
                    if ($image_info[2] === IMAGETYPE_PNG) {
                        echo '<div class="status success">✅ Image is PNG</div>';
                        
                        // Try to load with GD
                        if ($has_gd) {
                            $img = @imagecreatefrompng($watermark_path);
                            if ($img) {
                                echo '<div class="status success">✅ Successfully loaded with GD</div>';
                                
                                // Check for alpha
                                imagealphablending($img, false);
                                imagesavealpha($img, true);
                                echo '<div class="status success">✅ Alpha channel enabled</div>';
                                
                                imagedestroy($img);
                            } else {
                                echo '<div class="status error">❌ Failed to load with GD</div>';
                            }
                        }
                    } else {
                        echo '<div class="status warning">⚠️ Image is not PNG. Transparency may not work.</div>';
                    }
                } else {
                    echo '<div class="status error">❌ Cannot read image info. File may be corrupted.</div>';
                }
                
            } else {
                echo '<div class="status error">❌ File does NOT exist at path</div>';
                echo '<p><strong>Possible causes:</strong></p>';
                echo '<ul>';
                echo '<li>Image was deleted from media library</li>';
                echo '<li>URL to path conversion failed</li>';
                echo '<li>Incorrect upload directory structure</li>';
                echo '</ul>';
                echo '<p><strong>Solution:</strong> Re-upload the logo via Settings > Yo ToWebP</p>';
            }
            
            echo '</div>';
        }

        // Test 5: Debug Mode
        echo '<div class="test-group">';
        echo '<h2>Test 5: Debug Mode</h2>';
        
        $debug_mode = get_option('yo_towebp_debug_mode', 0);
        
        if ($debug_mode) {
            echo '<div class="status success">✅ Debug mode is enabled</div>';
            echo '<p>Check your error logs for detailed messages starting with "Yo ToWebP:"</p>';
            
            // Try to find error log location
            $error_log = ini_get('error_log');
            if ($error_log) {
                echo '<p><strong>Error log location:</strong></p>';
                echo '<code>' . esc_html($error_log) . '</code>';
            }
            
            // Common log locations
            echo '<p><strong>Common log locations to check:</strong></p>';
            echo '<ul>';
            echo '<li><code>/var/log/apache2/error.log</code> (Linux Apache)</li>';
            echo '<li><code>/var/log/nginx/error.log</code> (Linux Nginx)</li>';
            echo '<li><code>/Applications/MAMP/logs/php_error.log</code> (macOS MAMP)</li>';
            echo '<li><code>' . WP_CONTENT_DIR . '/debug.log</code> (WordPress)</li>';
            echo '</ul>';
            
            echo '<p><strong>View logs with:</strong></p>';
            echo '<pre>tail -f /path/to/error.log | grep "Yo ToWebP"</pre>';
            
        } else {
            echo '<div class="status warning">⚠️ Debug mode is disabled</div>';
            echo '<p>Enable it in Settings > Yo ToWebP to see detailed error messages.</p>';
        }
        
        echo '</div>';

        // Final recommendations
        echo '<div class="test-group">';
        echo '<h2>📋 Action Items</h2>';
        
        $issues = [];
        
        if (!$watermark_enabled) {
            $issues[] = 'Enable watermark in Settings > Yo ToWebP';
        }
        
        if ($watermark_type === 'image' && empty($watermark_image)) {
            $issues[] = 'Upload a logo image via "Upload Image/Logo" button';
        }
        
        if ($watermark_type === 'image' && !empty($watermark_image)) {
            $watermark_path = str_replace($base_url, $base_dir, $watermark_image);
            if (!file_exists($watermark_path)) {
                $issues[] = 'Re-upload logo - current file not found';
            }
        }
        
        if (!$has_gd && !$has_imagick) {
            $issues[] = 'Install GD or Imagick extension';
        }
        
        if (!$debug_mode) {
            $issues[] = 'Enable debug mode for troubleshooting';
        }
        
        if (empty($issues)) {
            echo '<div class="status success">✅ All checks passed! Your watermark should work.</div>';
            echo '<p><strong>Next step:</strong> Upload a test image via Media > Add New to verify watermark appears.</p>';
        } else {
            echo '<div class="status warning">⚠️ Please complete these actions:</div>';
            echo '<ol>';
            foreach ($issues as $issue) {
                echo '<li>' . esc_html($issue) . '</li>';
            }
            echo '</ol>';
        }
        
        echo '</div>';
        ?>

        <p style="text-align: center; margin-top: 40px;">
            <a href="<?php echo admin_url('options-general.php?page=yo-towebp'); ?>" class="button">Go to Settings</a>
            <a href="<?php echo admin_url('upload.php'); ?>" class="button">Go to Media Library</a>
        </p>

        <p style="text-align: center; color: #666; font-size: 13px; margin-top: 40px;">
            <strong>Note:</strong> Delete this file after testing for security reasons.
        </p>
    </div>
</body>
</html>
