<?php
/**
 * Plugin Name: Yo ToWebP Plus
 * Description: Automatically converts uploaded images to WebP format with cropping (16:9, 4:3) and watermark features.
 * Version: 2.0.0
 * Author: yowisben@live.com
 * License: GPL v2 or later
 */

if (!defined('ABSPATH')) {
    exit;
}

class YoToWebP {

    private $quality = 80;
    private $supported_formats = ['jpg', 'jpeg', 'png', 'gif'];

    public function __construct() {
        add_action('init', [$this, 'init']);
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
    }

    public function init() {
        if (!$this->is_webp_supported()) {
            add_action('admin_notices', [$this, 'webp_support_notice']);
            return;
        }

        add_filter('wp_handle_upload', [$this, 'handle_upload'], 10, 2);
        add_filter('wp_generate_attachment_metadata', [$this, 'generate_attachment_metadata'], 10, 2);
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'settings_init']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
    }
    
    public function enqueue_admin_assets($hook) {
        if ('settings_page_yo-towebp' !== $hook) {
            return;
        }
        
        // Enqueue WordPress media uploader
        wp_enqueue_media();
        
        wp_enqueue_style('yo-towebp-admin', plugins_url('assets/css/admin.css', __FILE__), [], '2.0.0');
        wp_enqueue_script('yo-towebp-admin', plugins_url('assets/js/admin.js', __FILE__), ['jquery'], '2.0.0', true);
    }

    public function activate() {
        add_option('yo_towebp_quality', 80);
        add_option('yo_towebp_delete_original', 1);
        add_option('yo_towebp_create_thumbnails', 1);
        add_option('yo_towebp_debug_mode', 0);
        
        // Crop settings
        add_option('yo_towebp_crop_enabled', 0);
        add_option('yo_towebp_crop_ratio', '16:9');
        add_option('yo_towebp_crop_position', 'center');
        
        // Watermark settings
        add_option('yo_towebp_watermark_enabled', 0);
        add_option('yo_towebp_watermark_type', 'text'); // 'text' or 'image'
        add_option('yo_towebp_watermark_text', 'Copyright © ' . get_bloginfo('name'));
        add_option('yo_towebp_watermark_image', ''); // Path to watermark image
        add_option('yo_towebp_watermark_position', 'bottom-right');
        add_option('yo_towebp_watermark_size', 3); // Percentage
        add_option('yo_towebp_watermark_size_type', 'width'); // 'width' or 'height'
        add_option('yo_towebp_watermark_opacity', 70);
        add_option('yo_towebp_watermark_color', '#ffffff');
    }

    private function debug_log($message) {
        if (get_option('yo_towebp_debug_mode', 0)) {
            error_log('Yo ToWebP: ' . $message);
        }
    }

    public function deactivate() {
        // Keep settings for future use
    }

    public function is_webp_supported() {
        if (extension_loaded('gd')) {
            return function_exists('imagewebp');
        }
        if (extension_loaded('imagick')) {
            return in_array('WEBP', Imagick::queryFormats());
        }
        return false;
    }

    public function webp_support_notice() {
        echo '<div class="notice notice-error"><p>';
        echo 'Yo ToWebP: WebP support is not available. Please enable GD extension with WebP support or Imagick.';
        echo '</p></div>';
    }

    public function handle_upload($upload, $context) {
        if (!isset($upload['file']) || !isset($upload['type'])) {
            $this->debug_log('Upload missing file or type');
            return $upload;
        }

        $file_path = $upload['file'];
        $file_info = pathinfo($file_path);
        $extension = strtolower($file_info['extension']);

        $this->debug_log('Processing upload: ' . $file_path . ' (type: ' . $extension . ')');

        if (!in_array($extension, $this->supported_formats)) {
            $this->debug_log('File format not supported: ' . $extension);
            return $upload;
        }

        $webp_path = $file_info['dirname'] . '/' . $file_info['filename'] . '.webp';

        $this->debug_log('Converting to WebP: ' . $webp_path);

        // Apply cropping if enabled
        if (get_option('yo_towebp_crop_enabled', 0)) {
            $this->debug_log('Applying crop before conversion');
            $file_path = $this->apply_crop($file_path, $extension);
        }

        if ($this->create_webp_image($file_path, $webp_path, $extension)) {
            $this->debug_log('WebP conversion successful');
            
            // Apply watermark if enabled
            if (get_option('yo_towebp_watermark_enabled', 0)) {
                $this->debug_log('Applying watermark');
                $this->apply_watermark($webp_path);
            }

            if (get_option('yo_towebp_delete_original', 1)) {
                if (unlink($file_path)) {
                    $this->debug_log('Original file deleted: ' . $file_path);
                } else {
                    $this->debug_log('Failed to delete original file: ' . $file_path);
                }
            }

            $upload['file'] = $webp_path;
            $upload['url'] = str_replace($file_info['basename'], $file_info['filename'] . '.webp', $upload['url']);
            $upload['type'] = 'image/webp';

            $this->debug_log('Upload metadata updated to WebP');
        } else {
            $this->debug_log('WebP conversion failed');
        }

        return $upload;
    }

    public function generate_attachment_metadata($metadata, $attachment_id) {
        $file_path = get_attached_file($attachment_id);
        $file_info = pathinfo($file_path);

        if ($file_info['extension'] !== 'webp') {
            return $metadata;
        }

        $upload_dir = wp_upload_dir();
        $sizes = [];

        $intermediate_sizes = get_intermediate_image_sizes();

        foreach ($intermediate_sizes as $size) {
            $size_data = $this->get_image_size_data($size);
            if ($size_data) {
                $thumbnail_path = $this->create_thumbnail_webp($file_path, $size_data['width'], $size_data['height'], $size);
                if ($thumbnail_path) {
                    $thumbnail_filename = basename($thumbnail_path);
                    $sizes[$size] = [
                        'file' => $thumbnail_filename,
                        'width' => $size_data['width'],
                        'height' => $size_data['height'],
                        'mime-type' => 'image/webp'
                    ];
                }
            }
        }

        if (!empty($sizes)) {
            $metadata['sizes'] = $sizes;
        }

        return $metadata;
    }

    public function get_image_size_data($size) {
        $size_data = wp_get_additional_image_sizes()[$size] ?? null;
        if (!$size_data) {
            switch ($size) {
                case 'thumbnail':
                    $size_data = ['width' => get_option('thumbnail_size_w'), 'height' => get_option('thumbnail_size_h')];
                    break;
                case 'medium':
                    $size_data = ['width' => get_option('medium_size_w'), 'height' => get_option('medium_size_h')];
                    break;
                case 'large':
                    $size_data = ['width' => get_option('large_size_w'), 'height' => get_option('large_size_h')];
                    break;
                case 'medium_large':
                    $size_data = ['width' => 768, 'height' => 0];
                    break;
            }
        }
        return $size_data;
    }

    public function create_thumbnail_webp($source_path, $width, $height, $size_name) {
        $file_info = pathinfo($source_path);

        list($orig_width, $orig_height) = getimagesize($source_path);

        if ($width == 0 && $height == 0) {
            return false;
        }

        if ($width == 0) {
            $width = intval(($orig_width * $height) / $orig_height);
        }
        if ($height == 0) {
            $height = intval(($orig_height * $width) / $orig_width);
        }

        $ratio = min($width / $orig_width, $height / $orig_height);
        $new_width = intval($orig_width * $ratio);
        $new_height = intval($orig_height * $ratio);

        $thumbnail_path = $file_info['dirname'] . '/' . $file_info['filename'] . '-' . $new_width . 'x' . $new_height . '.webp';

        if (extension_loaded('gd')) {
            return $this->resize_image_gd($source_path, $thumbnail_path, $new_width, $new_height) ? $thumbnail_path : false;
        } elseif (extension_loaded('imagick')) {
            return $this->resize_image_imagick($source_path, $thumbnail_path, $new_width, $new_height) ? $thumbnail_path : false;
        }

        return false;
    }

    public function create_webp_image($source_path, $dest_path, $source_format) {
        $quality = get_option('yo_towebp_quality', $this->quality);

        if (extension_loaded('gd') && function_exists('imagewebp')) {
            return $this->create_webp_with_gd($source_path, $dest_path, $source_format, $quality);
        } elseif (extension_loaded('imagick')) {
            return $this->create_webp_with_imagick($source_path, $dest_path, $quality);
        }

        return false;
    }

    private function create_webp_with_gd($source_path, $dest_path, $source_format, $quality) {
        switch ($source_format) {
            case 'jpg':
            case 'jpeg':
                $image = imagecreatefromjpeg($source_path);
                break;
            case 'png':
                $image = imagecreatefrompng($source_path);
                imagepalettetotruecolor($image);
                imagealphablending($image, true);
                imagesavealpha($image, true);
                break;
            case 'gif':
                $image = imagecreatefromgif($source_path);
                break;
            default:
                return false;
        }

        if (!$image) {
            return false;
        }

        $result = imagewebp($image, $dest_path, $quality);
        imagedestroy($image);

        return $result;
    }

    private function create_webp_with_imagick($source_path, $dest_path, $quality) {
        try {
            $imagick = new Imagick($source_path);
            $imagick->setImageFormat('webp');
            $imagick->setImageCompressionQuality($quality);
            $imagick->writeImage($dest_path);
            $imagick->destroy();
            return true;
        } catch (Exception $e) {
            return false;
        }
    }


    private function resize_image_gd($source_path, $dest_path, $width, $height) {
        $source = imagecreatefromwebp($source_path);
        if (!$source) return false;

        $orig_width = imagesx($source);
        $orig_height = imagesy($source);

        $ratio = min($width / $orig_width, $height / $orig_height);
        $new_width = intval($orig_width * $ratio);
        $new_height = intval($orig_height * $ratio);

        $resized = imagecreatetruecolor($new_width, $new_height);
        imagealphablending($resized, false);
        imagesavealpha($resized, true);

        imagecopyresampled($resized, $source, 0, 0, 0, 0, $new_width, $new_height, $orig_width, $orig_height);

        $result = imagewebp($resized, $dest_path, get_option('yo_towebp_quality', 80));

        imagedestroy($source);
        imagedestroy($resized);

        return $result;
    }

    private function resize_image_imagick($source_path, $dest_path, $width, $height) {
        try {
            $imagick = new Imagick($source_path);
            $imagick->thumbnailImage($width, $height, true);
            $imagick->setImageFormat('webp');
            $imagick->setImageCompressionQuality(get_option('yo_towebp_quality', 80));
            $imagick->writeImage($dest_path);
            $imagick->destroy();
            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    private function apply_crop($source_path, $extension) {
        $ratio = get_option('yo_towebp_crop_ratio', '16:9');
        $position = get_option('yo_towebp_crop_position', 'center');
        
        list($width, $height) = getimagesize($source_path);
        
        // Calculate target dimensions based on ratio
        $ratio_parts = explode(':', $ratio);
        $target_ratio = $ratio_parts[0] / $ratio_parts[1];
        $current_ratio = $width / $height;
        
        if (abs($current_ratio - $target_ratio) < 0.01) {
            // Already in correct ratio
            return $source_path;
        }
        
        // Calculate crop dimensions
        if ($current_ratio > $target_ratio) {
            // Image is wider, crop width
            $new_width = (int)($height * $target_ratio);
            $new_height = $height;
            $x_offset = $this->calculate_offset($width - $new_width, $position, true);
            $y_offset = 0;
        } else {
            // Image is taller, crop height
            $new_width = $width;
            $new_height = (int)($width / $target_ratio);
            $x_offset = 0;
            $y_offset = $this->calculate_offset($height - $new_height, $position, false);
        }
        
        $cropped_path = $source_path . '.cropped.tmp';
        
        if (extension_loaded('gd')) {
            $success = $this->crop_image_gd($source_path, $cropped_path, $extension, $x_offset, $y_offset, $new_width, $new_height);
        } elseif (extension_loaded('imagick')) {
            $success = $this->crop_image_imagick($source_path, $cropped_path, $x_offset, $y_offset, $new_width, $new_height);
        } else {
            return $source_path;
        }
        
        if ($success && file_exists($cropped_path)) {
            unlink($source_path);
            rename($cropped_path, $source_path);
        }
        
        return $source_path;
    }
    
    private function calculate_offset($difference, $position, $is_horizontal) {
        switch ($position) {
            case 'top':
            case 'left':
                return 0;
            case 'center':
                return (int)($difference / 2);
            case 'bottom':
            case 'right':
                return $difference;
            default:
                return (int)($difference / 2);
        }
    }
    
    private function crop_image_gd($source_path, $dest_path, $extension, $x, $y, $width, $height) {
        switch ($extension) {
            case 'jpg':
            case 'jpeg':
                $source = imagecreatefromjpeg($source_path);
                break;
            case 'png':
                $source = imagecreatefrompng($source_path);
                break;
            case 'gif':
                $source = imagecreatefromgif($source_path);
                break;
            default:
                return false;
        }
        
        if (!$source) return false;
        
        $cropped = imagecreatetruecolor($width, $height);
        
        // Preserve transparency for PNG
        if ($extension === 'png') {
            imagealphablending($cropped, false);
            imagesavealpha($cropped, true);
            $transparent = imagecolorallocatealpha($cropped, 0, 0, 0, 127);
            imagefill($cropped, 0, 0, $transparent);
        }
        
        imagecopyresampled($cropped, $source, 0, 0, $x, $y, $width, $height, $width, $height);
        
        $result = false;
        switch ($extension) {
            case 'jpg':
            case 'jpeg':
                $result = imagejpeg($cropped, $dest_path, 95);
                break;
            case 'png':
                $result = imagepng($cropped, $dest_path, 9);
                break;
            case 'gif':
                $result = imagegif($cropped, $dest_path);
                break;
        }
        
        imagedestroy($source);
        imagedestroy($cropped);
        
        return $result;
    }
    
    private function crop_image_imagick($source_path, $dest_path, $x, $y, $width, $height) {
        try {
            $imagick = new Imagick($source_path);
            $imagick->cropImage($width, $height, $x, $y);
            $imagick->writeImage($dest_path);
            $imagick->destroy();
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    private function apply_watermark($image_path) {
        $watermark_type = get_option('yo_towebp_watermark_type', 'text');
        $position = get_option('yo_towebp_watermark_position', 'bottom-right');
        $size_percent = get_option('yo_towebp_watermark_size', 3);
        $size_type = get_option('yo_towebp_watermark_size_type', 'width');
        $opacity = get_option('yo_towebp_watermark_opacity', 70);
        
        // Get image dimensions
        list($img_width, $img_height) = getimagesize($image_path);
        
        // Calculate base dimension for size calculation
        switch ($size_type) {
            case 'height':
                $base_dimension = $img_height;
                break;
            case 'min':
                $base_dimension = min($img_width, $img_height);
                break;
            case 'max':
                $base_dimension = max($img_width, $img_height);
                break;
            case 'width':
            default:
                $base_dimension = $img_width;
                break;
        }
        
        if ($watermark_type === 'image') {
            // Image watermark
            $watermark_image = get_option('yo_towebp_watermark_image', '');
            if (empty($watermark_image)) {
                $this->debug_log('Watermark image URL is empty');
                return false;
            }
            
            // Convert URL to file path
            $watermark_path = $this->url_to_path($watermark_image);
            if (!$watermark_path || !file_exists($watermark_path)) {
                $this->debug_log('Watermark image file not found: ' . $watermark_path);
                return false;
            }
            
            // Calculate watermark size
            $watermark_size = (int)(($base_dimension * $size_percent) / 100);
            $watermark_size = max(20, min($watermark_size, 500)); // Min 20px, max 500px
            
            if (extension_loaded('gd')) {
                return $this->apply_image_watermark_gd($image_path, $watermark_path, $position, $watermark_size, $opacity);
            } elseif (extension_loaded('imagick')) {
                return $this->apply_image_watermark_imagick($image_path, $watermark_path, $position, $watermark_size, $opacity);
            }
        } else {
            // Text watermark (no background box)
            $text = get_option('yo_towebp_watermark_text', '');
            if (empty($text)) {
                return false;
            }
            
            $color = get_option('yo_towebp_watermark_color', '#ffffff');
            
            // Calculate font size
            $font_size = (int)(($base_dimension * $size_percent) / 100);
            $font_size = max(10, min($font_size, 200));
            
            // Convert hex color to RGB
            $color = str_replace('#', '', $color);
            $r = hexdec(substr($color, 0, 2));
            $g = hexdec(substr($color, 2, 2));
            $b = hexdec(substr($color, 4, 2));
            
            if (extension_loaded('gd')) {
                return $this->apply_text_watermark_gd($image_path, $text, $position, $font_size, $opacity, $r, $g, $b);
            } elseif (extension_loaded('imagick')) {
                return $this->apply_text_watermark_imagick($image_path, $text, $position, $font_size, $opacity, $r, $g, $b);
            }
        }
        
        return false;
    }
    
    private function apply_text_watermark_gd($image_path, $text, $position, $size, $opacity, $r, $g, $b) {
        $image = imagecreatefromwebp($image_path);
        if (!$image) return false;
        
        $width = imagesx($image);
        $height = imagesy($image);
        
        // Calculate alpha (0-127 for GD, where 0 is opaque and 127 is transparent)
        $alpha = (int)(127 - ($opacity / 100 * 127));
        
        // Create color WITHOUT background - just transparent text color
        $color = imagecolorallocatealpha($image, $r, $g, $b, $alpha);
        
        // Try to use a TrueType font if available
        $font_file = null;
        $possible_fonts = [
            '/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf',
            '/System/Library/Fonts/Helvetica.ttc',
            '/usr/share/fonts/truetype/liberation/LiberationSans-Regular.ttf',
        ];
        
        foreach ($possible_fonts as $font) {
            if (file_exists($font)) {
                $font_file = $font;
                break;
            }
        }
        
        // Calculate text bounding box
        if ($font_file) {
            $bbox = imagettfbbox($size, 0, $font_file, $text);
            $text_width = abs($bbox[4] - $bbox[0]);
            $text_height = abs($bbox[5] - $bbox[1]);
        } else {
            // Fallback to built-in font
            $text_width = imagefontwidth(5) * strlen($text);
            $text_height = imagefontheight(5);
        }
        
        $padding = 10;
        list($x, $y) = $this->calculate_watermark_position($position, $width, $height, $text_width, $text_height, $padding);
        
        // Draw text WITHOUT background box
        if ($font_file) {
            imagettftext($image, $size, 0, $x, $y + $text_height, $color, $font_file, $text);
        } else {
            imagestring($image, 5, $x, $y, $text, $color);
        }
        
        $result = imagewebp($image, $image_path, get_option('yo_towebp_quality', 80));
        imagedestroy($image);
        
        return $result;
    }
    
    private function apply_text_watermark_imagick($image_path, $text, $position, $size, $opacity, $r, $g, $b) {
        try {
            $imagick = new Imagick($image_path);
            $draw = new ImagickDraw();
            
            // Set font properties WITHOUT background
            $draw->setFillColor(new ImagickPixel("rgba($r, $g, $b, " . ($opacity / 100) . ")"));
            $draw->setFontSize($size);
            $draw->setFont('Arial');
            
            // Get text dimensions
            $metrics = $imagick->queryFontMetrics($draw, $text);
            $text_width = $metrics['textWidth'];
            $text_height = $metrics['textHeight'];
            
            $width = $imagick->getImageWidth();
            $height = $imagick->getImageHeight();
            
            $padding = 10;
            list($x, $y) = $this->calculate_watermark_position($position, $width, $height, $text_width, $text_height, $padding);
            
            // Draw text WITHOUT background
            $imagick->annotateImage($draw, $x, $y + $text_height, 0, $text);
            $imagick->setImageFormat('webp');
            $imagick->writeImage($image_path);
            $imagick->destroy();
            
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    private function apply_image_watermark_gd($image_path, $watermark_path, $position, $watermark_size, $opacity) {
        $this->debug_log('=== Starting Image Watermark (GD) ===');
        $this->debug_log('Main image: ' . $image_path);
        $this->debug_log('Watermark: ' . $watermark_path);
        
        // Load main image
        $image = @imagecreatefromwebp($image_path);
        if (!$image) {
            $this->debug_log('ERROR: Failed to load main image');
            return false;
        }
        $this->debug_log('Main image loaded successfully');
        
        // Load watermark image
        $watermark = $this->load_image_gd($watermark_path);
        if (!$watermark) {
            $this->debug_log('ERROR: Failed to load watermark image');
            imagedestroy($image);
            return false;
        }
        $this->debug_log('Watermark image loaded successfully');
        
        $img_width = imagesx($image);
        $img_height = imagesy($image);
        $wm_width = imagesx($watermark);
        $wm_height = imagesy($watermark);
        
        $this->debug_log("Image dimensions: {$img_width}x{$img_height}");
        $this->debug_log("Watermark dimensions: {$wm_width}x{$wm_height}");
        
        // Resize watermark proportionally
        $ratio = $wm_width / $wm_height;
        $new_wm_width = $watermark_size;
        $new_wm_height = (int)($watermark_size / $ratio);
        
        $this->debug_log("Resizing watermark to: {$new_wm_width}x{$new_wm_height}");
        
        // Create resized watermark with transparency support
        $resized_wm = imagecreatetruecolor($new_wm_width, $new_wm_height);
        if (!$resized_wm) {
            $this->debug_log('ERROR: Failed to create resized watermark canvas');
            imagedestroy($image);
            imagedestroy($watermark);
            return false;
        }
        
        // Enable alpha blending for transparency
        imagealphablending($resized_wm, false);
        imagesavealpha($resized_wm, true);
        
        // Fill with transparent background
        $transparent = imagecolorallocatealpha($resized_wm, 0, 0, 0, 127);
        imagefill($resized_wm, 0, 0, $transparent);
        
        // Copy and resize watermark
        $resize_result = imagecopyresampled($resized_wm, $watermark, 0, 0, 0, 0, $new_wm_width, $new_wm_height, $wm_width, $wm_height);
        if (!$resize_result) {
            $this->debug_log('ERROR: Failed to resize watermark');
            imagedestroy($image);
            imagedestroy($watermark);
            imagedestroy($resized_wm);
            return false;
        }
        $this->debug_log('Watermark resized successfully');
        
        // Calculate position
        $padding = 10;
        list($x, $y) = $this->calculate_watermark_position($position, $img_width, $img_height, $new_wm_width, $new_wm_height, $padding);
        
        $this->debug_log("Watermark position: x={$x}, y={$y}, opacity={$opacity}");
        
        // Enable alpha blending on destination image
        imagealphablending($image, true);
        imagesavealpha($image, true);
        
        // Copy watermark with transparency and opacity
        $merge_result = $this->imagecopymerge_alpha($image, $resized_wm, $x, $y, 0, 0, $new_wm_width, $new_wm_height, $opacity);
        if (!$merge_result) {
            $this->debug_log('ERROR: Failed to merge watermark with image');
            imagedestroy($image);
            imagedestroy($watermark);
            imagedestroy($resized_wm);
            return false;
        }
        $this->debug_log('Watermark merged successfully');
        
        // Save as WebP
        $quality = get_option('yo_towebp_quality', 80);
        $result = @imagewebp($image, $image_path, $quality);
        
        // Clean up
        imagedestroy($image);
        imagedestroy($watermark);
        imagedestroy($resized_wm);
        
        if ($result) {
            $this->debug_log('=== Image saved successfully with watermark ===');
        } else {
            $this->debug_log('ERROR: Failed to save image');
        }
        
        return $result;
    }
    
    private function apply_image_watermark_imagick($image_path, $watermark_path, $position, $watermark_size, $opacity) {
        try {
            $imagick = new Imagick($image_path);
            $watermark = new Imagick($watermark_path);
            
            // Get dimensions
            $img_width = $imagick->getImageWidth();
            $img_height = $imagick->getImageHeight();
            $wm_width = $watermark->getImageWidth();
            $wm_height = $watermark->getImageHeight();
            
            // Resize watermark proportionally
            $ratio = $wm_width / $wm_height;
            $new_wm_width = $watermark_size;
            $new_wm_height = (int)($watermark_size / $ratio);
            
            $watermark->resizeImage($new_wm_width, $new_wm_height, Imagick::FILTER_LANCZOS, 1);
            
            // Apply opacity
            $watermark->evaluateImage(Imagick::EVALUATE_MULTIPLY, $opacity / 100, Imagick::CHANNEL_ALPHA);
            
            // Calculate position
            $padding = 10;
            list($x, $y) = $this->calculate_watermark_position($position, $img_width, $img_height, $new_wm_width, $new_wm_height, $padding);
            
            // Composite watermark onto image
            $imagick->compositeImage($watermark, Imagick::COMPOSITE_OVER, $x, $y);
            $imagick->setImageFormat('webp');
            $imagick->writeImage($image_path);
            
            $imagick->destroy();
            $watermark->destroy();
            
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    private function load_image_gd($path) {
        $this->debug_log('Loading image: ' . $path);
        
        if (!file_exists($path)) {
            $this->debug_log('ERROR: File does not exist');
            return false;
        }
        
        if (!is_readable($path)) {
            $this->debug_log('ERROR: File is not readable');
            return false;
        }
        
        $info = @getimagesize($path);
        if (!$info) {
            $this->debug_log('ERROR: Cannot get image info - file may be corrupted');
            return false;
        }
        
        $this->debug_log('Image type: ' . image_type_to_mime_type($info[2]));
        
        $image = false;
        switch ($info[2]) {
            case IMAGETYPE_PNG:
                $this->debug_log('Loading PNG...');
                $image = @imagecreatefrompng($path);
                if ($image) {
                    // Preserve transparency for PNG
                    imagealphablending($image, false);
                    imagesavealpha($image, true);
                    $this->debug_log('PNG loaded with alpha channel preserved');
                } else {
                    $this->debug_log('ERROR: Failed to load PNG');
                }
                break;
                
            case IMAGETYPE_JPEG:
                $this->debug_log('Loading JPEG...');
                $image = @imagecreatefromjpeg($path);
                if ($image) {
                    $this->debug_log('JPEG loaded successfully');
                } else {
                    $this->debug_log('ERROR: Failed to load JPEG');
                }
                break;
                
            case IMAGETYPE_GIF:
                $this->debug_log('Loading GIF...');
                $image = @imagecreatefromgif($path);
                if ($image) {
                    // Preserve transparency for GIF
                    imagealphablending($image, false);
                    imagesavealpha($image, true);
                    $this->debug_log('GIF loaded with transparency preserved');
                } else {
                    $this->debug_log('ERROR: Failed to load GIF');
                }
                break;
                
            case IMAGETYPE_WEBP:
                $this->debug_log('Loading WebP...');
                if (function_exists('imagecreatefromwebp')) {
                    $image = @imagecreatefromwebp($path);
                    if ($image) {
                        imagealphablending($image, false);
                        imagesavealpha($image, true);
                        $this->debug_log('WebP loaded with alpha channel preserved');
                    } else {
                        $this->debug_log('ERROR: Failed to load WebP');
                    }
                } else {
                    $this->debug_log('ERROR: imagecreatefromwebp function not available');
                }
                break;
                
            default:
                $this->debug_log('ERROR: Unsupported image type: ' . $info[2]);
                return false;
        }
        
        if (!$image) {
            $this->debug_log('ERROR: Failed to create image resource');
        }
        
        return $image;
    }
    
    private function url_to_path($url) {
        // Get WordPress upload directory
        $upload_dir = wp_upload_dir();
        $base_url = $upload_dir['baseurl'];
        $base_dir = $upload_dir['basedir'];
        
        // Convert URL to path
        if (strpos($url, $base_url) === 0) {
            $path = str_replace($base_url, $base_dir, $url);
            return $path;
        }
        
        // If URL is already a path, return it
        if (file_exists($url)) {
            return $url;
        }
        
        return false;
    }
    
    private function imagecopymerge_alpha($dst_im, $src_im, $dst_x, $dst_y, $src_x, $src_y, $src_w, $src_h, $pct) {
        $pct /= 100;
        $w = imagesx($src_im);
        $h = imagesy($src_im);
        
        imagealphablending($src_im, false);
        
        $minalpha = 127;
        for ($x = 0; $x < $w; $x++) {
            for ($y = 0; $y < $h; $y++) {
                $alpha = (imagecolorat($src_im, $x, $y) >> 24) & 0xFF;
                if ($alpha < $minalpha) {
                    $minalpha = $alpha;
                }
            }
        }
        
        for ($x = 0; $x < $w; $x++) {
            for ($y = 0; $y < $h; $y++) {
                $colorxy = imagecolorat($src_im, $x, $y);
                $alpha = ($colorxy >> 24) & 0xFF;
                if ($minalpha !== 127) {
                    $alpha = 127 + 127 * $pct * ($alpha - 127) / (127 - $minalpha);
                } else {
                    $alpha += 127 * $pct;
                }
                $alphacolorxy = imagecolorallocatealpha(
                    $src_im,
                    ($colorxy >> 16) & 0xFF,
                    ($colorxy >> 8) & 0xFF,
                    $colorxy & 0xFF,
                    $alpha
                );
                if (!imagesetpixel($src_im, $x, $y, $alphacolorxy)) {
                    return false;
                }
            }
        }
        
        imagecopy($dst_im, $src_im, $dst_x, $dst_y, $src_x, $src_y, $src_w, $src_h);
        return true;
    }
    
    private function calculate_watermark_position($position, $img_width, $img_height, $text_width, $text_height, $padding) {
        $x = $padding;
        $y = $padding;
        
        switch ($position) {
            case 'top-left':
                $x = $padding;
                $y = $padding;
                break;
            case 'top-center':
                $x = ($img_width - $text_width) / 2;
                $y = $padding;
                break;
            case 'top-right':
                $x = $img_width - $text_width - $padding;
                $y = $padding;
                break;
            case 'middle-left':
                $x = $padding;
                $y = ($img_height - $text_height) / 2;
                break;
            case 'center':
                $x = ($img_width - $text_width) / 2;
                $y = ($img_height - $text_height) / 2;
                break;
            case 'middle-right':
                $x = $img_width - $text_width - $padding;
                $y = ($img_height - $text_height) / 2;
                break;
            case 'bottom-left':
                $x = $padding;
                $y = $img_height - $text_height - $padding;
                break;
            case 'bottom-center':
                $x = ($img_width - $text_width) / 2;
                $y = $img_height - $text_height - $padding;
                break;
            case 'bottom-right':
            default:
                $x = $img_width - $text_width - $padding;
                $y = $img_height - $text_height - $padding;
                break;
        }
        
        return [(int)$x, (int)$y];
    }

    public function add_admin_menu() {
        add_options_page(
            'Yo ToWebP Settings',
            'Yo ToWebP',
            'manage_options',
            'yo-towebp',
            [$this, 'options_page']
        );
    }

    public function settings_init() {
        register_setting('yo_towebp', 'yo_towebp_quality');
        register_setting('yo_towebp', 'yo_towebp_delete_original');
        register_setting('yo_towebp', 'yo_towebp_create_thumbnails');
        register_setting('yo_towebp', 'yo_towebp_debug_mode');
        
        // Crop settings
        register_setting('yo_towebp', 'yo_towebp_crop_enabled');
        register_setting('yo_towebp', 'yo_towebp_crop_ratio');
        register_setting('yo_towebp', 'yo_towebp_crop_position');
        
        // Watermark settings
        register_setting('yo_towebp', 'yo_towebp_watermark_enabled');
        register_setting('yo_towebp', 'yo_towebp_watermark_type');
        register_setting('yo_towebp', 'yo_towebp_watermark_text');
        register_setting('yo_towebp', 'yo_towebp_watermark_image');
        register_setting('yo_towebp', 'yo_towebp_watermark_position');
        register_setting('yo_towebp', 'yo_towebp_watermark_size');
        register_setting('yo_towebp', 'yo_towebp_watermark_size_type');
        register_setting('yo_towebp', 'yo_towebp_watermark_opacity');
        register_setting('yo_towebp', 'yo_towebp_watermark_color');

        add_settings_section(
            'yo_towebp_section',
            'Conversion Settings',
            [$this, 'settings_section_callback'],
            'yo_towebp'
        );

        add_settings_field(
            'yo_towebp_quality',
            'WebP Quality (1-100)',
            [$this, 'quality_render'],
            'yo_towebp',
            'yo_towebp_section'
        );

        add_settings_field(
            'yo_towebp_delete_original',
            'Delete Original Files',
            [$this, 'delete_original_render'],
            'yo_towebp',
            'yo_towebp_section'
        );

        add_settings_field(
            'yo_towebp_create_thumbnails',
            'Create Thumbnails',
            [$this, 'create_thumbnails_render'],
            'yo_towebp',
            'yo_towebp_section'
        );

        add_settings_field(
            'yo_towebp_debug_mode',
            'Debug Mode',
            [$this, 'debug_mode_render'],
            'yo_towebp',
            'yo_towebp_section'
        );
        
        // Crop section
        add_settings_section(
            'yo_towebp_crop_section',
            'Image Cropping',
            [$this, 'crop_section_callback'],
            'yo_towebp'
        );
        
        add_settings_field(
            'yo_towebp_crop_enabled',
            'Enable Auto Crop',
            [$this, 'crop_enabled_render'],
            'yo_towebp',
            'yo_towebp_crop_section'
        );
        
        add_settings_field(
            'yo_towebp_crop_ratio',
            'Crop Ratio',
            [$this, 'crop_ratio_render'],
            'yo_towebp',
            'yo_towebp_crop_section'
        );
        
        add_settings_field(
            'yo_towebp_crop_position',
            'Crop Position',
            [$this, 'crop_position_render'],
            'yo_towebp',
            'yo_towebp_crop_section'
        );
        
        // Watermark section
        add_settings_section(
            'yo_towebp_watermark_section',
            'Watermark Settings',
            [$this, 'watermark_section_callback'],
            'yo_towebp'
        );
        
        add_settings_field(
            'yo_towebp_watermark_enabled',
            'Enable Watermark',
            [$this, 'watermark_enabled_render'],
            'yo_towebp',
            'yo_towebp_watermark_section'
        );
        
        add_settings_field(
            'yo_towebp_watermark_type',
            'Watermark Type',
            [$this, 'watermark_type_render'],
            'yo_towebp',
            'yo_towebp_watermark_section'
        );
        
        add_settings_field(
            'yo_towebp_watermark_text',
            'Watermark Text',
            [$this, 'watermark_text_render'],
            'yo_towebp',
            'yo_towebp_watermark_section'
        );
        
        add_settings_field(
            'yo_towebp_watermark_image',
            'Watermark Image',
            [$this, 'watermark_image_render'],
            'yo_towebp',
            'yo_towebp_watermark_section'
        );
        
        add_settings_field(
            'yo_towebp_watermark_position',
            'Watermark Position',
            [$this, 'watermark_position_render'],
            'yo_towebp',
            'yo_towebp_watermark_section'
        );
        
        add_settings_field(
            'yo_towebp_watermark_size',
            'Watermark Size',
            [$this, 'watermark_size_render'],
            'yo_towebp',
            'yo_towebp_watermark_section'
        );
        
        add_settings_field(
            'yo_towebp_watermark_size_type',
            'Size Based On',
            [$this, 'watermark_size_type_render'],
            'yo_towebp',
            'yo_towebp_watermark_section'
        );
        
        add_settings_field(
            'yo_towebp_watermark_opacity',
            'Watermark Opacity',
            [$this, 'watermark_opacity_render'],
            'yo_towebp',
            'yo_towebp_watermark_section'
        );
        
        add_settings_field(
            'yo_towebp_watermark_color',
            'Watermark Color',
            [$this, 'watermark_color_render'],
            'yo_towebp',
            'yo_towebp_watermark_section'
        );
    }

    public function quality_render() {
        $quality = get_option('yo_towebp_quality', 80);
        echo '<input type="range" id="yo_towebp_quality" name="yo_towebp_quality" value="' . $quality . '" min="1" max="100">';
        echo '<span class="range-value" id="quality-value">' . $quality . '</span>';
        echo '<p class="description">Higher values mean better quality but larger file sizes.</p>';
    }

    public function delete_original_render() {
        $delete = get_option('yo_towebp_delete_original', 1);
        echo '<input type="checkbox" name="yo_towebp_delete_original" value="1"' . checked(1, $delete, false) . '>';
        echo '<p class="description">Delete original image files after conversion to save disk space.</p>';
    }

    public function create_thumbnails_render() {
        $thumbnails = get_option('yo_towebp_create_thumbnails', 1);
        echo '<input type="checkbox" name="yo_towebp_create_thumbnails" value="1"' . checked(1, $thumbnails, false) . '>';
        echo '<p class="description">Generate WebP thumbnails for different image sizes.</p>';
    }

    public function debug_mode_render() {
        $debug = get_option('yo_towebp_debug_mode', 0);
        echo '<input type="checkbox" name="yo_towebp_debug_mode" value="1"' . checked(1, $debug, false) . '>';
        echo '<p class="description">Enable debug logging to WordPress error log for troubleshooting.</p>';
    }
    
    public function crop_section_callback() {
        echo '<p>Automatically crop uploaded images to a specific aspect ratio.</p>';
    }
    
    public function crop_enabled_render() {
        $enabled = get_option('yo_towebp_crop_enabled', 0);
        echo '<input type="checkbox" id="yo_towebp_crop_enabled" name="yo_towebp_crop_enabled" value="1"' . checked(1, $enabled, false) . '>';
        echo '<p class="description">Enable automatic cropping for uploaded images.</p>';
    }
    
    public function crop_ratio_render() {
        $ratio = get_option('yo_towebp_crop_ratio', '16:9');
        echo '<div class="crop-settings">';
        echo '<select name="yo_towebp_crop_ratio">';
        echo '<option value="16:9"' . selected('16:9', $ratio, false) . '>16:9 (Widescreen)</option>';
        echo '<option value="4:3"' . selected('4:3', $ratio, false) . '>4:3 (Standard)</option>';
        echo '<option value="1:1"' . selected('1:1', $ratio, false) . '>1:1 (Square)</option>';
        echo '<option value="21:9"' . selected('21:9', $ratio, false) . '>21:9 (Ultrawide)</option>';
        echo '</select>';
        echo '<p class="description">Choose the aspect ratio for cropping.</p>';
        echo '</div>';
    }
    
    public function crop_position_render() {
        $position = get_option('yo_towebp_crop_position', 'center');
        echo '<div class="crop-settings">';
        echo '<select name="yo_towebp_crop_position">';
        echo '<option value="center"' . selected('center', $position, false) . '>Center</option>';
        echo '<option value="top"' . selected('top', $position, false) . '>Top</option>';
        echo '<option value="bottom"' . selected('bottom', $position, false) . '>Bottom</option>';
        echo '<option value="left"' . selected('left', $position, false) . '>Left</option>';
        echo '<option value="right"' . selected('right', $position, false) . '>Right</option>';
        echo '</select>';
        echo '<p class="description">Choose which part of the image to keep when cropping.</p>';
        echo '</div>';
    }
    
    public function watermark_section_callback() {
        echo '<p>Add a watermark to your images automatically.</p>';
    }
    
    public function watermark_enabled_render() {
        $enabled = get_option('yo_towebp_watermark_enabled', 0);
        echo '<input type="checkbox" id="yo_towebp_watermark_enabled" name="yo_towebp_watermark_enabled" value="1"' . checked(1, $enabled, false) . '>';
        echo '<p class="description">Enable automatic watermarking for uploaded images.</p>';
    }
    
    public function watermark_type_render() {
        $type = get_option('yo_towebp_watermark_type', 'text');
        echo '<div class="watermark-settings">';
        echo '<select id="yo_towebp_watermark_type" name="yo_towebp_watermark_type">';
        echo '<option value="text"' . selected('text', $type, false) . '>Text Watermark</option>';
        echo '<option value="image"' . selected('image', $type, false) . '>Image/Logo Watermark</option>';
        echo '</select>';
        echo '<p class="description">Choose between text or image/logo watermark.</p>';
        echo '</div>';
    }
    
    public function watermark_text_render() {
        $text = get_option('yo_towebp_watermark_text', 'Copyright © ' . get_bloginfo('name'));
        echo '<div class="watermark-settings watermark-text-settings">';
        echo '<input type="text" id="yo_towebp_watermark_text" name="yo_towebp_watermark_text" value="' . esc_attr($text) . '" style="width: 400px;">';
        echo '<p class="description">The text to display as watermark (no background box).</p>';
        echo '</div>';
    }
    
    public function watermark_image_render() {
        $image_url = get_option('yo_towebp_watermark_image', '');
        echo '<div class="watermark-settings watermark-image-settings">';
        echo '<input type="text" id="yo_towebp_watermark_image" name="yo_towebp_watermark_image" value="' . esc_attr($image_url) . '" style="width: 400px;" readonly>';
        echo '<button type="button" class="button" id="upload_watermark_image">Upload Image/Logo</button>';
        if ($image_url) {
            echo '<button type="button" class="button" id="remove_watermark_image" style="margin-left: 5px;">Remove</button>';
            echo '<div class="watermark-image-preview">';
            echo '<img src="' . esc_url($image_url) . '" alt="Watermark preview" style="max-width: 200px; max-height: 100px; display: block; margin-top: 10px; border: 1px solid #ddd; padding: 5px; background: repeating-conic-gradient(#e0e0e0 0% 25%, #ffffff 0% 50%) 50% / 20px 20px;">';
            echo '</div>';
        }
        echo '<p class="description">Upload a logo or image (PNG with transparency recommended, max 500KB).</p>';
        echo '</div>';
    }
    
    public function watermark_position_render() {
        $position = get_option('yo_towebp_watermark_position', 'bottom-right');
        $positions = [
            'top-left' => 'Top Left',
            'top-center' => 'Top Center',
            'top-right' => 'Top Right',
            'middle-left' => 'Middle Left',
            'center' => 'Center',
            'middle-right' => 'Middle Right',
            'bottom-left' => 'Bottom Left',
            'bottom-center' => 'Bottom Center',
            'bottom-right' => 'Bottom Right',
        ];
        
        echo '<div class="watermark-settings">';
        echo '<div class="watermark-position-grid">';
        foreach ($positions as $value => $label) {
            $checked = ($position === $value) ? 'checked' : '';
            echo '<label class="watermark-position-btn' . ($position === $value ? ' active' : '') . '">';
            echo '<input type="radio" name="yo_towebp_watermark_position" value="' . $value . '" ' . $checked . ' style="display:none;">';
            echo '<span>' . esc_html($label) . '</span>';
            echo '</label>';
        }
        echo '</div>';
        echo '<p class="description">Choose where to place the watermark.</p>';
        echo '</div>';
    }
    
    public function watermark_size_render() {
        $size = get_option('yo_towebp_watermark_size', 3);
        echo '<div class="watermark-settings">';
        echo '<input type="range" id="yo_towebp_watermark_size" name="yo_towebp_watermark_size" value="' . $size . '" min="1" max="20" step="0.5">';
        echo '<span class="range-value" id="watermark-size-value">' . $size . '%</span>';
        echo '<p class="description">Watermark size as percentage (1-20%).</p>';
        echo '</div>';
    }
    
    public function watermark_size_type_render() {
        $size_type = get_option('yo_towebp_watermark_size_type', 'width');
        echo '<div class="watermark-settings">';
        echo '<select id="yo_towebp_watermark_size_type" name="yo_towebp_watermark_size_type">';
        echo '<option value="width"' . selected('width', $size_type, false) . '>Based on Image Width</option>';
        echo '<option value="height"' . selected('height', $size_type, false) . '>Based on Image Height</option>';
        echo '<option value="min"' . selected('min', $size_type, false) . '>Based on Smaller Dimension</option>';
        echo '<option value="max"' . selected('max', $size_type, false) . '>Based on Larger Dimension</option>';
        echo '</select>';
        echo '<p class="description">Choose which dimension to use for calculating watermark size.</p>';
        echo '</div>';
    }
    
    public function watermark_opacity_render() {
        $opacity = get_option('yo_towebp_watermark_opacity', 70);
        echo '<div class="watermark-settings">';
        echo '<input type="range" id="yo_towebp_watermark_opacity" name="yo_towebp_watermark_opacity" value="' . $opacity . '" min="10" max="100">';
        echo '<span class="range-value" id="watermark-opacity-value">' . $opacity . '</span>';
        echo '<p class="description">Opacity of the watermark (10-100).</p>';
        echo '</div>';
    }
    
    public function watermark_color_render() {
        $color = get_option('yo_towebp_watermark_color', '#ffffff');
        echo '<div class="watermark-settings">';
        echo '<input type="color" id="yo_towebp_watermark_color" name="yo_towebp_watermark_color" value="' . esc_attr($color) . '">';
        echo '<span style="margin-left: 10px;">' . esc_html($color) . '</span>';
        echo '<p class="description">Color of the watermark text.</p>';
        echo '</div>';
    }

    public function settings_section_callback() {
        echo '<p>Configure how images are converted to WebP format.</p>';
    }

    public function options_page() {
        ?>
        <div class="wrap yo-towebp-settings">
            <h1>Yo ToWebP Settings</h1>
            
            <form action="options.php" method="post">
                <?php
                settings_fields('yo_towebp');
                do_settings_sections('yo_towebp');
                submit_button();
                ?>
            </form>

            <div class="yo-towebp-section">
                <h2>Watermark Preview</h2>
                <div class="watermark-preview">
                    <img src="<?php echo plugins_url('assets/images/preview.jpg', __FILE__); ?>" 
                         alt="Preview" 
                         class="watermark-preview-image"
                         onerror="this.style.display='none'; this.nextElementSibling.style.display='block';">
                    <div style="display: none; padding: 40px; background: #f0f0f0; text-align: center;">
                        Preview image not available
                    </div>
                    <div class="watermark-preview-text">Sample Watermark</div>
                    <img class="watermark-preview-logo" 
                         src="<?php echo esc_url(get_option('yo_towebp_watermark_image', '')); ?>" 
                         alt="Logo Watermark"
                         style="display: none; position: absolute; pointer-events: none;">
                </div>
                <p class="description">This preview shows how your watermark will appear on images.</p>
            </div>

            <div class="yo-towebp-section">
                <h2>System Information</h2>
                <table class="wp-list-table widefat fixed striped">
                    <tr>
                        <td><strong>GD Extension</strong></td>
                        <td><?php echo extension_loaded('gd') ? '✅ Available' : '❌ Not Available'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>WebP Support (GD)</strong></td>
                        <td><?php echo (extension_loaded('gd') && function_exists('imagewebp')) ? '✅ Available' : '❌ Not Available'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Imagick Extension</strong></td>
                        <td><?php echo extension_loaded('imagick') ? '✅ Available' : '❌ Not Available'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>WebP Support (Imagick)</strong></td>
                        <td><?php echo (extension_loaded('imagick') && in_array('WEBP', Imagick::queryFormats())) ? '✅ Available' : '❌ Not Available'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Crop Feature</strong></td>
                        <td><?php echo get_option('yo_towebp_crop_enabled', 0) ? '✅ Enabled' : '❌ Disabled'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Watermark Feature</strong></td>
                        <td><?php echo get_option('yo_towebp_watermark_enabled', 0) ? '✅ Enabled' : '❌ Disabled'; ?></td>
                    </tr>
                </table>
            </div>
        </div>
        <?php
    }
}

new YoToWebP();